<?php
/**
 * Created by PhpStorm.
 * User: yz520
 * Date: 2020/4/8
 * Time: 4:57 PM
 */

namespace kmkj\oss\core;

use AlibabaCloud\Client\AlibabaCloud;
use kmkj\oss\client\TstClient;
use kmkj\oss\ex\OssException;
use kmkj\oss\header\OssConst;

/**
 * @link https://help.aliyun.com/document_detail/66053.html?spm=a2c4g.11186623.2.11.6e2a39afnLF9gf#reference-sdg-3pv-xdb
 * Class Sts
 * @package kmkj\oss\core
 */
class Sts extends TstClient
{

    private $expireTime = 900;

    protected $roleArn ='';

    public function setExpireTime($timeOut)
    {
        if (!is_numeric($timeOut)) {
            throw new OssException("过期时间只能是数字");
        }
        $this->expireTime = $timeOut;
        return $this;
    }
    /**
     * @param $action string 授权读写权限
     * @param $path string 授权路径
     * @link 代码编辑器  https://api.aliyun.com/#/?product=Sts&version=2015-04-01&api=AssumeRole&params={}&tab=DEMO&lang=PHP
     * @link 授权策略编辑器 http://gosspublic.alicdn.com/ram-policy-editor/index.html?spm=a2c4g.11186623.2.10.5b8a3eb1BYXNqB
     * @throws \AlibabaCloud\Client\Exception\ClientException
     * @throws \AlibabaCloud\Client\Exception\ServerException
     * @throws \kmkj\oss\ex\OssException
     */
    public function createTempUser($action,$path='*'){


        $request = $this->getClient();
        $policy = <<<json
{
  "Version": "1",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
       {$action}
      ],
      "Resource": [
        "acs:oss:*:*:{$this->getBucket()}/{$path}"
      ],
      "Condition": {}
    }
  ]
}
json;


        $options =[
            'query' => [
                'RegionId' => $this->getRegionId(),
                'RoleArn' =>  $this->getRoleArn(),//'acs:ram::1848687233876471:role/oss',
                'RoleSessionName' => $this->getRoleName(),
                'DurationSeconds' => $this->expireTime,
                'Policy' => $policy,
            ]
        ];

        $request->client($this->getName());
        $result = $request->options($options)->request();
        $res = $result->toArray();

        if (empty($res['Credentials'])) {
            throw new OssException("获取临时token失败");
        }
        return $res['Credentials'];
    }
    
    /**
     * @param $contentType
     * @param $size
     * @return array
     * @author 阿豹
     */
    protected function createPolicy($contentType,$size=null)
    {
        // 设置过期时间，默认1小时
        // 必须使用gmdate生成UTC时间，格式末尾的Z表示UTC时区，确保全球服务器签名一致性
        $expiration = gmdate('Y-m-d\TH:i:s\Z', time() + $this->expireTime);

        // 如果contentType为null，使用图片集合
        if (empty($contentType)) {
            $contentTypes = OssConst::$CONTENT_TYPES_IMAGE;
        } else {
            // 支持字符串或数组
            $contentTypes = is_array($contentType) ? $contentType : [$contentType];
        }

        // 构建policy对象
        $policy = [
            'expiration' => $expiration,
            'conditions' => [
                ['in', '$content-type', $contentTypes]
            ]
        ];

        // 如果指定了size，添加文件大小限制
        if ($size !== null) {
            array_unshift($policy['conditions'], ['content-length-range', 1, $size]);
        }

        return $policy;
    }
    /**
     * 创建V4版本的policy
     * @param string $contentType 内容类型，null时使用图片集合
     * @param int|null $size 文件大小限制
     * @param string $date ISO 8601日期格式 例如：20231203T121212Z
     * @param string $credential credential字符串
     * @param string|null $securityToken 安全令牌（STS使用）
     * @return array policy数组
     */
    protected function createPolicyV4($contentType, $size, $date, $credential, $securityToken = null)
    {
        // 设置过期时间，默认1小时
        // 必须使用gmdate生成UTC时间，格式末尾的Z表示UTC时区，确保全球服务器签名一致性
        $expiration = gmdate('Y-m-d\TH:i:s.000\Z', time() + $this->expireTime);

        // 如果contentType为null，使用图片集合
        if (empty($contentType)) {
            $contentTypes = OssConst::$CONTENT_TYPES_IMAGE;
        } else {
            // 支持字符串或数组
            $contentTypes = is_array($contentType) ? $contentType : [$contentType];
        }

        // 构建policy对象
        $policy = [
            'expiration' => $expiration,
            'conditions' => [
                ['bucket' => $this->getBucket()],
                ['x-oss-signature-version' => 'OSS4-HMAC-SHA256'],
                ['x-oss-credential' => $credential],
                ['x-oss-date' => $date],
                ['in', '$content-type', $contentTypes]
            ]
        ];

        // 如果使用STS，添加security token
        if ($securityToken !== null) {
            $policy['conditions'][] = ['x-oss-security-token' => $securityToken];
        }

        // 如果指定了size，添加文件大小限制
        if ($size !== null) {
            array_unshift($policy['conditions'], ['content-length-range', 1, $size]);
        }

        return $policy;
    }

    /**
     * 计算V4签名的SigningKey
     * @param string $accessKeySecret 访问密钥
     * @param string $dateShort 日期，格式：YYYYMMDD
     * @param string $region 区域ID
     * @return string 签名密钥
     */
    protected function getSigningKeyV4($accessKeySecret, $dateShort, $region)
    {
        // DateKey = HMAC-SHA256("aliyun_v4" + AccessKeySecret, Date)
        $dateKey = hash_hmac('sha256', $dateShort, 'aliyun_v4' . $accessKeySecret, true);

        // DateRegionKey = HMAC-SHA256(DateKey, Region)
        $dateRegionKey = hash_hmac('sha256', $region, $dateKey, true);

        // DateRegionServiceKey = HMAC-SHA256(DateRegionKey, "oss")
        $dateRegionServiceKey = hash_hmac('sha256', 'oss', $dateRegionKey, true);

        // SigningKey = HMAC-SHA256(DateRegionServiceKey, "aliyun_v4_request")
        $signingKey = hash_hmac('sha256', 'aliyun_v4_request', $dateRegionServiceKey, true);

        return $signingKey;
    }

    /**
     * 获取V4版本的POST签名
     * @param string $path 上传路径
     * @param string|array|null $contentType 内容类型，null时使用图片集合
     * @param int|null $size 文件大小限制（字节）
     * @return array 返回签名信息
     * @return array{
     *     policy: string,
     *     signature: string,
     *     accessKeyId: string,
     *     securityToken: string,
     *     expiration: string,
     *     date: string,
     *     credential: string,
     *     url: string,
     *     formData: array{
     *         x-oss-signature-version: string,
     *         x-oss-credential: string,
     *         x-oss-date: string,
     *         x-oss-signature: string,
     *         x-oss-security-token: string
     *     }
     * }
     * @throws \Exception
     * @author 阿豹
     */
    public function createPostSignV4($path,$contentType=null,$size=null)
    {
        
        
        // 1. 获取临时凭证
        $credentials = $this->createTempUser(OssConst::STS_POLICY_ACTION_ALL, $path);

        // 2. 生成日期和时间
        $now = time();
        $date = gmdate('Ymd\THis\Z', $now); // ISO 8601格式：20231203T121212Z
        $dateShort = gmdate('Ymd', $now); // 短日期格式：20231203
        $region = $this->getRegionId();

        // 3. 构造credential字符串
        // 格式：<AccessKeyId>/<date>/<region>/oss/aliyun_v4_request
        $credential = $credentials['AccessKeyId'] . '/' . $dateShort . '/' . $region . '/oss/aliyun_v4_request';

        // 4. 构建policy对象
        $policy = $this->createPolicyV4($contentType, $size, $date, $credential, $credentials['SecurityToken']);

        // 5. 将policy转为JSON字符串（UTF-8编码）
        $policyString = json_encode($policy, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);

        // 6. Base64编码policy，作为待签名字符串（StringToSign）
        $policyBase64 = base64_encode($policyString);

        // 7. 计算SigningKey
        $signingKey = $this->getSigningKeyV4($credentials['AccessKeySecret'], $dateShort, $region);

        // 8. 使用SigningKey对Base64编码的policy进行HMAC-SHA256签名
        // 将二进制哈希值转换为十六进制形式
        $signature = hash_hmac('sha256', $policyBase64, $signingKey);

        // 9. 构造返回结果，formData包含前端POST表单需要的所有字段
        $formData = [
            'x-oss-signature-version' => 'OSS4-HMAC-SHA256',
            'x-oss-credential' => $credential,
            'x-oss-date' => $date,
            'x-oss-signature' => $signature,
            'x-oss-security-token' => $credentials['SecurityToken']
        ];

        // 10. 构造OSS三级域名URL
        $endpoint = preg_replace('#^https?://#', '', $this->getEndpoint());
        $url = 'https://' . $this->getBucket() . '.' . $endpoint;

        return [
            'policy' => $policyBase64,
            'signature' => $signature,
            'accessKeyId' => $credentials['AccessKeyId'],
            'securityToken' => $credentials['SecurityToken'],
            'expiration' => $credentials['Expiration'],
            'date' => $date,
            'credential' => $credential,
            'url' => $url,
            'formData' => $formData
        ];
    }
    
    /**
     * @deprecated
     * 获取V1版本的POST签名
     * @param string $path 上传路径
     * @param string|array|null $contentType 内容类型，null时使用图片集合
     * @param int|null $size 文件大小限制（字节）
     * @return array 返回签名信息
     * @return array{
     *     policy: string,
     *     signature: string,
     *     accessKeyId: string,
     *     securityToken: string,
     *     expiration: string,
     *     url: string,
     *     formData: array{
     *         policy: string,
     *         OSSAccessKeyId: string,
     *         Signature: string,
     *         x-oss-security-token: string
     *     }
     * }
     * @throws \Exception
     * @author 阿豹
     */
    public function createPostSign($path,$contentType=null,$size=null)
    {
        // 1. 获取临时凭证
        $credentials = $this->createTempUser(OssConst::STS_POLICY_ACTION_ALL, $path);

        // 2. 构建policy对象
        $policy = $this->createPolicy($contentType, $size);

        // 3. 将policy转为JSON字符串（UTF-8编码）
        $policyString = json_encode($policy, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);

        // 4. Base64编码policy，作为待签名字符串（StringToSign）
        $policyBase64 = base64_encode($policyString);

        // 5. 使用临时凭证的AccessKeySecret对Base64编码的policy进行HMAC-SHA1签名
        // Signature = base64(hmac-sha1(AccessKeySecret, base64(policy)))
        $signature = base64_encode(hash_hmac('sha1', $policyBase64, $credentials['AccessKeySecret'], true));

        // 6. 构造返回结果，formData包含前端POST表单需要的所有字段
        $formData = [
            'policy' => $policyBase64,
            'OSSAccessKeyId' => $credentials['AccessKeyId'],
            'Signature' => $signature,
            'x-oss-security-token' => $credentials['SecurityToken']
        ];

        // 7. 构造OSS三级域名URL
        $endpoint = preg_replace('#^https?://#', '', $this->getEndpoint());
        $url = 'https://' . $this->getBucket() . '.' . $endpoint;

        return [
            'policy' => $policyBase64,
            'signature' => $signature,
            'accessKeyId' => $credentials['AccessKeyId'],
            'securityToken' => $credentials['SecurityToken'],
            'expiration' => $credentials['Expiration'],
            'url' => $url,
            'formData' => $formData
        ];
    }
}